/*
 * Copyright (C) 2024 Igalia S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#if !defined(__WEBKIT_H_INSIDE__) && !defined(BUILDING_WEBKIT)
#error "Only <webkit/webkit.h> can be included directly."
#endif

#ifndef WebKitWebExtensionMatchPattern_h
#define WebKitWebExtensionMatchPattern_h

#include <glib-object.h>
#include <gio/gio.h>
#include <webkit/WebKitDefines.h>

G_BEGIN_DECLS

/**
 * WebKitWebExtensionMatchPatternOptions:
 * @WEBKIT_WEB_EXTENSION_MATCH_PATTERN_OPTIONS_NONE: No special matching options.
 * @WEBKIT_WEB_EXTENSION_MATCH_PATTERN_OPTIONS_IGNORE_SCHEMES: The scheme components should be ignored while matching.
 * @WEBKIT_WEB_EXTENSION_MATCH_PATTERN_OPTIONS_IGNORE_PATHS: The host components should be ignored while matching.
 * @WEBKIT_WEB_EXTENSION_MATCH_PATTERN_OPTIONS_MATCH_BIDIRECTIONALLY: Two patterns should be checked in either direction while matching (A matches B, or B matches A). Invalid for matching URLs.
 *
 * Enum values representing matching options.
 *
 * Since: 2.48
 */
typedef enum {
    WEBKIT_WEB_EXTENSION_MATCH_PATTERN_OPTIONS_NONE = 1 << 0,
    WEBKIT_WEB_EXTENSION_MATCH_PATTERN_OPTIONS_IGNORE_SCHEMES = 1 << 1,
    WEBKIT_WEB_EXTENSION_MATCH_PATTERN_OPTIONS_IGNORE_PATHS = 1 << 2,
    WEBKIT_WEB_EXTENSION_MATCH_PATTERN_OPTIONS_MATCH_BIDIRECTIONALLY = 1 << 3,
} WebKitWebExtensionMatchPatternOptions;

#define WEBKIT_TYPE_WEB_EXTENSION_MATCH_PATTERN            (webkit_web_extension_match_pattern_get_type ())

typedef struct _WebKitWebExtensionMatchPattern WebKitWebExtensionMatchPattern;

WEBKIT_API GType
webkit_web_extension_match_pattern_get_type (void);

WEBKIT_API WebKitWebExtensionMatchPattern *
webkit_web_extension_match_pattern_ref (WebKitWebExtensionMatchPattern *matchPattern);

WEBKIT_API void
webkit_web_extension_match_pattern_unref (WebKitWebExtensionMatchPattern *matchPattern);

WEBKIT_API WebKitWebExtensionMatchPattern*
webkit_web_extension_match_pattern_new_all_urls (void);

WEBKIT_API WebKitWebExtensionMatchPattern*
webkit_web_extension_match_pattern_new_all_hosts_and_schemes (void);

WEBKIT_API WebKitWebExtensionMatchPattern*
webkit_web_extension_match_pattern_new_with_string (const gchar  *string,
                                                    GError      **error);

WEBKIT_API WebKitWebExtensionMatchPattern*
webkit_web_extension_match_pattern_new_with_scheme (const gchar  *scheme,
                                                    const gchar  *host,
                                                    const gchar  *path,
                                                    GError      **error);

WEBKIT_API void
webkit_web_extension_match_pattern_register_custom_url_scheme (const gchar *urlScheme);

WEBKIT_DEPRECATED_FOR(webkit_web_extension_match_pattern_register_custom_url_scheme) void
webkit_web_extension_match_pattern_register_custom_URL_scheme (const gchar *urlScheme);

WEBKIT_API const gchar *
webkit_web_extension_match_pattern_get_string (WebKitWebExtensionMatchPattern *matchPattern);

WEBKIT_API const gchar *
webkit_web_extension_match_pattern_get_scheme (WebKitWebExtensionMatchPattern *matchPattern);

WEBKIT_API const gchar *
webkit_web_extension_match_pattern_get_host (WebKitWebExtensionMatchPattern *matchPattern);

WEBKIT_API const gchar *
webkit_web_extension_match_pattern_get_path (WebKitWebExtensionMatchPattern *matchPattern);

WEBKIT_API gboolean
webkit_web_extension_match_pattern_get_matches_all_urls (WebKitWebExtensionMatchPattern *matchPattern);

WEBKIT_API gboolean
webkit_web_extension_match_pattern_get_matches_all_hosts (WebKitWebExtensionMatchPattern *matchPattern);

WEBKIT_API gboolean
webkit_web_extension_match_pattern_matches_url (WebKitWebExtensionMatchPattern        *matchPattern,
                                                const gchar                           *url,
                                                WebKitWebExtensionMatchPatternOptions  options);

WEBKIT_API gboolean
webkit_web_extension_match_pattern_matches_pattern (WebKitWebExtensionMatchPattern        *matchPattern,
                                                    WebKitWebExtensionMatchPattern        *pattern,
                                                    WebKitWebExtensionMatchPatternOptions  options);

G_DEFINE_AUTOPTR_CLEANUP_FUNC(WebKitWebExtensionMatchPattern, webkit_web_extension_match_pattern_unref)

G_END_DECLS

#endif
