/*
 * Copyright (C) 2025 Igalia S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#if !defined(__WEBKIT_H_INSIDE__) && !defined(BUILDING_WEBKIT)
#error "Only <webkit/webkit.h> can be included directly."
#endif

#ifndef WebKitXRPermissionRequest_h
#define WebKitXRPermissionRequest_h

#include <glib-object.h>
#include <webkit/WebKitDefines.h>
#include <webkit/WebKitSecurityOrigin.h>

G_BEGIN_DECLS

#define WEBKIT_TYPE_XR_PERMISSION_REQUEST            (webkit_xr_permission_request_get_type())

WEBKIT_DECLARE_FINAL_TYPE (WebKitXRPermissionRequest, webkit_xr_permission_request, WEBKIT, XR_PERMISSION_REQUEST, GObject)

/**
 * WebKitXRSessionMode:
 * @WEBKIT_XR_SESSION_MODE_INLINE: inline session mode.
 * @WEBKIT_XR_SESSION_MODE_IMMERSIVE_VR: immersive-vr session mode.
 * @WEBKIT_XR_SESSION_MODE_IMMERSIVE_AR: immersive-ar session mode.
 *
 * Enum values representing the XR session mode.
 *
 * See <https://immersive-web.github.io/webxr/#xrsessionmode-enum>
 *
 * Since: 2.52
 */
typedef enum {
    WEBKIT_XR_SESSION_MODE_INLINE,
    WEBKIT_XR_SESSION_MODE_IMMERSIVE_VR,
    WEBKIT_XR_SESSION_MODE_IMMERSIVE_AR,
} WebKitXRSessionMode;

/**
 * WebKitXRSessionFeatures:
 * @WEBKIT_XR_SESSION_FEATURES_VIEWER: viewer feature.
 * @WEBKIT_XR_SESSION_FEATURES_LOCAL: local feature.
 * @WEBKIT_XR_SESSION_FEATURES_LOCAL_FLOOR: local-floor feature.
 * @WEBKIT_XR_SESSION_FEATURES_BOUNDED_FLOOR: bounded-floor feature.
 * @WEBKIT_XR_SESSION_FEATURES_UNBOUNDED: unbounded feature.
 * @WEBKIT_XR_SESSION_FEATURES_HAND_TRACKING: hand tracking feature.
 *
 * Enum values representing the XR session features.
 *
 * See <https://immersive-web.github.io/webxr/#xrreferencespace-interface>.
 * And, see <https://immersive-web.github.io/webxr-hand-input/> for hand tracking.
 *
 * Since: 2.52
 */
typedef enum {
    WEBKIT_XR_SESSION_FEATURES_VIEWER        = 1 << 0,
    WEBKIT_XR_SESSION_FEATURES_LOCAL         = 1 << 1,
    WEBKIT_XR_SESSION_FEATURES_LOCAL_FLOOR   = 1 << 2,
    WEBKIT_XR_SESSION_FEATURES_BOUNDED_FLOOR = 1 << 3,
    WEBKIT_XR_SESSION_FEATURES_UNBOUNDED     = 1 << 4,
    WEBKIT_XR_SESSION_FEATURES_HAND_TRACKING = 1 << 5,
} WebKitXRSessionFeatures;

WEBKIT_API WebKitSecurityOrigin *
webkit_xr_permission_request_get_security_origin             (WebKitXRPermissionRequest *request);

WEBKIT_API WebKitXRSessionMode
webkit_xr_permission_request_get_session_mode                (WebKitXRPermissionRequest *request);

WEBKIT_API WebKitXRSessionFeatures
webkit_xr_permission_request_get_granted_features            (WebKitXRPermissionRequest *request) G_GNUC_WARN_UNUSED_RESULT;

WEBKIT_API WebKitXRSessionFeatures
webkit_xr_permission_request_get_consent_required_features   (WebKitXRPermissionRequest *request) G_GNUC_WARN_UNUSED_RESULT;

WEBKIT_API WebKitXRSessionFeatures
webkit_xr_permission_request_get_consent_optional_features   (WebKitXRPermissionRequest *request) G_GNUC_WARN_UNUSED_RESULT;

WEBKIT_API WebKitXRSessionFeatures
webkit_xr_permission_request_get_required_features_requested (WebKitXRPermissionRequest *request) G_GNUC_WARN_UNUSED_RESULT;

WEBKIT_API WebKitXRSessionFeatures
webkit_xr_permission_request_get_optional_features_requested (WebKitXRPermissionRequest *request) G_GNUC_WARN_UNUSED_RESULT;

WEBKIT_API void
webkit_xr_permission_request_set_granted_optional_features   (WebKitXRPermissionRequest *request,
                                                              WebKitXRSessionFeatures    granted);

G_END_DECLS

#endif
